#!/usr/bin/env ruby
# coding: utf-8

# Author: Christoffer Müller Madsen
# Version: 1.0

# TODO: Find løsning til "TAB-problem" i "Brugerinformations"-dialogboksen

require 'shellwords'

DIALOG="dialog --stdout"
SUPPORTED_LANGUAGES=[:en, :da]

env_lang = (ENV['LANG'] or "")[0..1].to_sym
LANG=if not env_lang.empty? then
       if SUPPORTED_LANGUAGES.include? env_lang
       then env_lang
       else :en
       end
     end

DEFAULT_PPD = { :MACOS => "/System/Library/Frameworks/ApplicationServices.framework/Frameworks/PrintCore.framework/Resources/Generic.ppd",
                :LINUX => "#{File.dirname __FILE__}/ppd/Generic-PostScript_Printer-Postscript.ppd" }

class Printer
  def initialize(tag, name, smb, location, ppd=DEFAULT_PPD[get_os])
    @tag = tag
    @name = name
    @smb = smb
    @location = location
    @ppd = ppd
  end
  
  attr_accessor :tag, :name, :smb, :location, :ppd
end

# Add color support for String objects
class String
  def colorise ansi_code
    "\e[#{ansi_code}m#{self}\e[0m"
  end

  def red
    colorise 31
  end

  def green
    colorise 32
  end

  def yellow
    colorise 33
  end

  def blue
    colorise 34
  end
end

class DependencyMissing < StandardError
end

# Define methods

def get_os
  case `uname`.strip
  when 'Linux'
    :LINUX
  when 'Darwin'
    :MACOS
  end
end

def get_unsafe_string s
  ((STRINGS[s] or {})[LANG] or
   ("MISSING_I18N: " + s.to_s + "+" + LANG.to_s))
end
  
def get_string s
  Shellwords.escape(get_unsafe_string s)
end

def print_status s
  puts " - " + s
end

def check_dependencies
  `which dialog`
  if $? != 0 then
    puts get_unsafe_string(:error_dependency_dialog)
    raise DependencyMissing, "Missing 'dialog'"
  end

  `which lpadmin`
  if $? != 0 then
    puts get_unsafe_string(:error_dependency_lpadmin)
    raise DependencyMissing, "Missing 'lpadmin'"
  end
end

def ask_for_printer_selection
  dialog_command = "#{DIALOG} --title #{get_string(:title_printer_selection)} --checklist #{get_string(:printer_selection_desc)} #{10 + PRINTERS.length} 50 18 "
  dialog_command << PRINTERS.map { |obj| "#{obj.tag} '#{obj.location}' 0" }.join(" ")
  res = `#{dialog_command}`
  res.split(" ").map { |s| PRINTERS_HASH[s] }
end

def wants_authentication?
  dialog_command = "#{DIALOG} --title #{get_string(:title_wants_authentication)} --defaultno --yesno #{get_string(:wants_authentication_desc)} 10 50"

  system dialog_command
  res = $?
  system "clear"
  
  if res == 0 then true else false end
end

def ask_authentication_info
  authinfo = []
  auth_good = false
  while auth_good != true
    dialog_command = "#{DIALOG} --title #{get_string(:title_authentication_info)} --insecure --mixedform #{get_string(:authentication_info_desc)} 10 48 2\
                       #{get_string(:username_label)}: 1 0 'UNI\\au' 1 11 30 30 0\
                       #{get_string(:password_label)}: 2 0 '' 2 11 30 30 1"
    res = `#{dialog_command}`
    if $? == 0 then
      authinfo = res.split("\n")
      if authinfo.length != 2 or
        authinfo[0].empty? or
        authinfo[1].empty? then
        `#{DIALOG} --title #{get_string(:title_auth_not_filled)} --msgbox #{get_string(:msg_auth_not_filled)} 8 30`
      else
        auth_good = true
      end
    else
      return nil
    end
  end
  
  authinfo
end

def ask_user_for_confirmation printer_list
  confirmation_list = printer_list.map { |obj| "* #{obj.tag}" }.join(" \n")
  dialog_command = "#{DIALOG} --title #{get_string(:title_confirmation)} --yesno #{get_string(:printer_confirmation_desc)}'\n\n#{confirmation_list}' #{8 + printer_list.length} 40"

  system dialog_command
  res = $?
  system "clear"

  if res == 0 then true else false end
end

def is_installed? printer  
  `lpstat -E -v`.include? printer.name
end

def install_printer printer, authinfo=nil
  puts "[#{printer.tag.blue}]"

  print_status get_unsafe_string(:status_check_already_setup)
  if is_installed? printer then
    return :EXISTS
  end

  print_status get_unsafe_string(:status_check_ppd_exists)
  if not File.file? printer.ppd then
    puts printer.ppd
    return :PPD_MISSING
  end
  
  print_status get_unsafe_string(:status_registering_printer)
  `lpadmin -p '#{printer.name}' -E -P '#{printer.ppd}' -L '#{printer.location}' -v '#{smb_prefix(printer.smb, authinfo)}'`
  
  if $? == 0 then
    :SUCCESS
  else
    :FAIL
  end
end

def smb_prefix s, authinfo=nil
  if authinfo then
    "smb://#{authinfo[0]}:#{authinfo[1]}@print.uni.au.dk/" + s
  else
    "smb://print.uni.au.dk/" + s
  end
end


# Define printers
require_relative 'printers.rb'

# Load strings
require_relative 'strings.rb'

# --------------------------------

# Check dependency for dialog
check_dependencies

# Add title to dialog
DIALOG << " --backtitle #{get_string(:dialog_title)}"

# Ask for printers to be installed
to_be_installed = ask_for_printer_selection

if to_be_installed.empty? then
  system "clear"
  puts get_unsafe_string(:stdout_no_printers_selected)
  exit 0
end

# Ask for authentication info
user_wants_authentication = wants_authentication?
authinfo = nil
if user_wants_authentication then
  authinfo = ask_authentication_info
end

# Confirm printers to be installed
confirmed = ask_user_for_confirmation to_be_installed

if confirmed then
  puts "#{get_unsafe_string(:stdout_printers_to_be_setup)}: #{ to_be_installed.map(&:tag).map(&:blue).join(", ") }"
  print "\n"
else
  puts get_unsafe_string(:not_confirmed)
  exit 0
end

# Install printers
fail_list = []
to_be_installed.each do |printer|
  case install_printer printer, authinfo
  when :SUCCESS
    puts (get_unsafe_string(:printer_setup_success) % printer.tag).green
    print "\n"
  when :FAIL
    puts (get_unsafe_string(:printer_setup_fail) % printer.tag).red
    print "\n"
    fail_list << printer
  when :EXISTS
    puts (get_unsafe_string(:printer_setup_exists) % printer.tag).yellow
    print "\n"
  when :PPD_MISSING
    puts (get_unsafe_string(:printer_setup_ppd_missing) % printer.tag).red
    print "\n"
    fail_list << printer
  end
end

if not fail_list.empty? then
  puts "\n#{get_unsafe_string(:post_printer_setup_failures)}".red
  exit 1
else
  exit 0
end
